---
url: /holoviews
name: HoloViews
description: HoloViews integration
icon: holoviews.svg
languages:
    - python
---

# ![](/assets/images/language_icons/matlab_50px.svg) HoloViews Overview

HoloViews is an ambitious project that aims to provide a flexible grammar of
visualization types and plot interactions. HoloViews specifications can be
displayed using a variety of technologies, including Plotly.js and Dash.

While HoloViews can be used to create a large variety of visualizations, for Dash
users it is particularly helpful for two use cases: Automatically linking selections
across multiple figures and displaying large data sets using Datashader.

HoloViews also provides a uniform interface to a variety of data structures,
making it easy to start out by visualizing small pandas DataFrames and then scale
up to GPU accelerated RAPIDS cudf DataFrames, or larger than memory Dask DataFrames.

For more background information, see the main HoloViews documentation at
https://holoviews.org/

## HoloViews Elements and Containers

The visualization primitives in HoloViews are called elements. Elements
in HoloViews are analogous to Plotly traces, and there are specific elements for
[`Scatter`](http://holoviews.org/reference/elements/plotly/Scatter.html#elements-plotly-gallery-scatter),
[`Area`](http://holoviews.org/reference/elements/plotly/Area.html#elements-plotly-gallery-area),
[`Bars`](http://holoviews.org/reference/elements/plotly/Bars.html#elements-plotly-gallery-bars),
[`Histogram`](http://holoviews.org/reference/elements/plotly/Histogram.html#elements-plotly-gallery-histogram),
[`Heatmap`](http://holoviews.org/reference/elements/plotly/HeatMap.html#elements-plotly-gallery-heatmap),
etc.

Elements can be grouped together into various containers including the
[`Overlay`](http://holoviews.org/reference/containers/plotly/Overlay.html#containers-plotly-gallery-overlay)
container for overlaying multiple compatible elements on the same axes, and the
[`Layout`](http://holoviews.org/reference/containers/plotly/Layout.html#containers-plotly-gallery-layout)
container for displaying multiple elements side by side as separate subplots.
Additionally, HoloViews supports several more advanced "Dimensioned" containers
to aid in the visualization of multi-dimensional datasets including the
[`HoloMap`](http://holoviews.org/reference/containers/plotly/HoloMap.html#containers-plotly-gallery-holomap),
[`Gridspace`](http://holoviews.org/reference/containers/plotly/GridSpace.html#containers-plotly-gallery-gridspace),
[`NdLayout`](http://holoviews.org/reference/containers/plotly/NdLayout.html#containers-plotly-gallery-ndlayout),
and
[`Ndoverlay`](http://holoviews.org/reference/containers/plotly/NdOverlay.html#containers-plotly-gallery-ndoverlay)
containers.

Finally, the
[`DynamicMap`](https://holoviews.org/reference/containers/plotly/DynamicMap.html#containers-plotly-gallery-dynamicmap)
is a special container than produces elements dynamically, often
in response to user interaction events. This documentation page does not discuss
the creation of general `DynamicMap` instances, but it's helpful to understand
that the `datashade` and `link_selections` transformations discussed below both
produce either `DynamicMap` instances, or containers of `DynamicMap` instances.

## HoloViews Datasets

While it's possible to build HoloViews elements directly from external data
structures like numpy arrays and pandas DataFrames, HoloViews also provides a
`Dataset` class that aims to serve as a universal interface to these data
structures. The recommended workflow is to first wrap the original data structure
(e.g. the pandas DataFrame) in a `Dataset` instance, and then construct
elements using this `Dataset`.

This workflow has two main advantages:

1. It makes it easy to swap out data structures in the future. For example, you
could develop a visualization using a small pandas DataFrame and then later
switch to a GPU accelerated cuDF DataFrame or a larger-than-memory dask
DataFrame.

2. It allows HoloViews to associate each visualization element with all of the
dimensions (i.e. columns in the case of a DataFrame) in the original `Dataset`.
This is what makes it possible for HoloViews to automate the process of
linking selections across visualizations that do not all display the same
dimensions. See the following sections for some examples of using the
`link_selections` function to accomplish this.

The examples in this documentation page use `Dataset` instances that wrap tabular
data structures. But Datasets also support wrapping gridded datasets like numpy
[`ndarray`](https://numpy.org/doc/stable/reference/generated/numpy.ndarray.html)
and xarray
[`DataArray`](http://xarray.pydata.org/en/stable/generated/xarray.DataArray.html)
objects. See the
[Tabular Datasets](http://holoviews.org/user_guide/Tabular_Datasets.html)
documentation page for more information on wrapping tabular data structures, and
see the
[Gridded Datasets](http://holoviews.org/user_guide/Gridded_Datasets.html)
documentation page for more information on wrapping gridded data structures.


## Building Dash Components from HoloViews Objects

HoloViews elements and containers can be converted into Dash components using the
`holoviews.plotting.plotly.dash.to_dash` function. This function inputs
a `dash.Dash` app instance and a list of HoloViews objects, and returns a
`namedtuple` with the following properties.

- `graphs`: This is a list of converted `Graph` components with the same
length as the input list of HoloViews objects. By default, these will have
type `dash_core_components.Graph`, but an alternative graph component class
(e.g. `dash_design_kit.Graph`) can be specified using the `graph_class`
argument to the `to_dash` function.
- `resets`: If the `reset_button=True` argument is passed to
`to_dash`, the `resets` property will hold a length 1 list
containing a Dash component that represents a reset button. When clicked,
this button will reset the graphs to their initial state. This will reset
both the figure viewports and other interactive states like the active
selection produced by the `link_selection` examples below. If
`reset_button=False`, the default, then this list will be empty.
- `store`: A Dash [`Store`](/dash-core-components/store) component that is used
internally to maintain the joint interactive state of all of the converted
Dash components.
- `kdims`: For Dimensioned HoloViews containers, the `kdims`
property holds a dictionary from key-dimension names to Dash components
that represent sliders for each key dimension. Dimensioned Containers are
not discussed further here; see the
[Dimensioned Containers](http://holoviews.org/user_guide/Dimensioned_Containers.html)
section in the HoloViews documentation for more information.
- `children`: This is a list that contains all of the components above. This can
be assigned directly to the `children` property of an `html.Div` component if
no additional layout structure is needed.

After calling `to_dash`, each of the resulting components must be
included somewhere in the app's layout. This can be done by *either*:

- including the `children` list as the `children` property of an `html.Div`
component.
- including all of the components stored in `graphs`, `resets`, `kdims`, and
`store` somewhere in the app's layout.


## Display Simple HoloViews Elements with Dash

This example loads the iris dataset included in plotly.py and wraps it in a
HoloViews `Dataset`. This `Dataset` is then used to construct a
[`Scatter`](http://holoviews.org/reference/elements/plotly/Scatter.html#elements-plotly-gallery-scatter)
element and a
[`Histogram`](http://holoviews.org/reference/elements/plotly/Histogram.html#elements-plotly-gallery-histogram)
element.  The `Histogram` element is created using the
[`histogram`](http://holoviews.org/Reference_Manual/holoviews.operation.html#holoviews.operation.histogram)
operation which is what executes the histogram binning algorithm.

These two elements are converted into two Dash `Graph` components using the
`to_dash` function, and are placed into a `Div` component along with the
associated `Store` component.

{{example holoviews-scatter}}

## Styling Figures Produced by HoloViews

There are two general approaches that can be used to customize the appearance
of the Plotly figures produced by HoloViews.

### Options System
The first is the HoloViews options system. This approach uses the syntax:

```python
element.opts(option1=value1, option2=value2)
```

This is very analogous to the `fig.update()` syntax that is used to update Plotly
`plotly.graph_object.Figure` objects. The available options for a particular
element type can be discovered from a Python or IPython REPL using the
`holoviews.help` function. For example:

```python
import holoviews as hv
hv.extension("plotly")
hv.help(hv.Scatter)
```

A slightly different syntax is used to style elements inside a container. Here is
an example of how `Scatter` element options would be applied to a container that
contains or produces `Scatter` elements:

```python
from holoviews import opts
container.opts(opts.Scatter(option1=value1, option2=value2))
```

This makes it possible to target options to elements of specific types within
a container. This is the approach that must be used to apply options to the
`DynamicMap` instances produced by the `datashade` and `link_selections`
transformations discussed below.

The example below customizes the appearance of a `Scatter` element using the
`size` and `color` options.

See the
[Applying Customizations](http://holoviews.org/user_guide/Applying_Customizations.html)
section of the HoloViews documentation for more information on styling figures
usng the options system.

### Plot Hooks
HoloViews aims to expose the most common plot options through the `opts` syntax
above, but the coverage of possible plotly configuration options is not exhaustive.
HoloViews provides a system called "plot hooks" to make it possible to apply arbitrary
figure customizations.  Every element has a special option named `hooks` that may
be set to a list of functions that should be applied to the figure that HoloViews
generates before it is displayed.

The example below uses a plot hook to change the default drag tool from
`zoom` to `pan`.

See the
[Plot hooks](http://holoviews.org/user_guide/Customizing_Plots.html#Plot-hooks)
section in the HoloViews documentation for more information.

{{example holoviews-scatter-styled}}

## Linked Selections with HoloViews

One HoloViews feature that is particularly convenient for Dash users is the ability
to automatically link selections across figures without the need to manually define
any callback functions.

This can be done by first creating a `link_selections` instance
(called `selection_linker` in the examples below) using the
`link_selections.instance()` method, and then calling this object as a function
with the elements or containers to be linked.

When these linked elements are passed to the `to_dash` function, Dash
callbacks to achieve this interactive linking behavior are automatically generated
and registered with the provided `dash.Dash` app instance.

This example shows how the `reset_button=True` argument to `to_dash` can
be used to create a Dash button component. When this button is clicked, the plot
viewport and selection states are reset to their original values.

For more background on linked selections in Holoviews, see the
[Linked Brushing](http://holoviews.org/user_guide/Linked_Brushing.html)
documentation section.

Try using the box-selection tool to select regions of space in each figure and
notice how the selection of the corresponding data is displayed in both figures.
Note that only box selection is supported right now. Lasso selection support
is not yet implemented.

{{example link-selections}}

## Visualizing Large Datasets with Datashader
Another HoloViews feature that is particularly convenient for Dash users is the
integration with [Datashader](https://datashader.org/).

Datashader is a Python library for quickly creating a variety of principled
visualizations of large datasets.

While the Plotly `scattergl` trace can handle hundreds of thousands of points,
Datashader can handle tens to hundreds of millions. The difference is that rather
than passing the entire dataset from the Python server to the browser for rendering,
Datashader rasterizes the dataset to a heatmap or image, and only this heatmap or
image is transferred to the browser for rendering.

To effectively use Datashader in an interactive context, it's necessary to rerender
the dataset each time the figure viewport changes. This can be accomplished in
Dash by installing a callback function that listens for changes to the
`relayoutData` prop.  Because of how HoloViews packages data lazily (without
rendering it immediately), replaying this pipeline of transformations can be
accomplished without manually defining any callbacks, making Datashader much easier
to use than if invoked without HoloViews.

This example loads the iris dataset included in plotly.py and then duplicates
it many times with added noise to generate a DataFrame with 1.5 million rows.
This large pandas DataFrame is wrapped in a HoloViews `Dataset` and then used to
construct a `Scatter` element.

The `datashade` operation is used to transform the `Scatter` element into
a datashaded scatter element that automatically updates in response to zoom / pan
events. The `to_dash` function is then used to build a single Dash
`Graph` component and a reset button.

When zooming and panning on this figure, notice how the datashaded image is
automatically updated. The reset button can be used to reset to the initial figure
viewport.

For more information on using datashader through HoloViews, see the
[Large Data](http://holoviews.org/user_guide/Large_Data.html) section of the
HoloViews documentation.

```python
from dash import Dash, html
from plotly.data import iris

import holoviews as hv
from holoviews.plotting.plotly.dash import to_dash
from holoviews.operation.datashader import datashade

import numpy as np
import pandas as pd

# Load iris dataset and replicate with noise to create large dataset
df_original = iris()[["sepal_length", "sepal_width", "petal_length", "petal_width"]]
df = pd.concat([
df_original + np.random.randn(*df_original.shape) * 0.1
for i in range(10000)
])
dataset = hv.Dataset(df)

scatter = datashade(
hv.Scatter(dataset, kdims=["sepal_length"], vdims=["sepal_width"])
).opts(title="Datashader with %d points" % len(dataset))

app = Dash()
components = to_dash(
app, [scatter], reset_button=True
)

app.layout = html.Div(components.children)

if __name__ == "__main__":
    app.run(debug=True)
```

![Demo Of Datashader with 1500000 points](/assets/images/integration/holoviews/datashader.gif "")

## Combining Datashader and Linked Selections

This examples shows how the two previous examples can be combined to support
linking selections across a histogram and a datashaded scatter plot of 1.5 million
points.

When using the box-selection tool to select regions of space in each figure,
notice how the selection of the corresponding data is displayed in both figures.
Also, when zooming and panning on datashaded scatter figure, notice how the
datashaded image is automatically updated. The reset button can be used to reset
to the initial figure viewport and clear the current selection.

```python
from dash import Dash, html
from plotly.data import iris

import holoviews as hv
from holoviews import opts
from holoviews.plotting.plotly.dash import to_dash
from holoviews.operation.datashader import datashade

import numpy as np
import pandas as pd

# Load iris dataset and replicate with noise to create large dataset
df_original = iris()[["sepal_length", "sepal_width", "petal_length", "petal_width"]]
df = pd.concat([
df_original + np.random.randn(*df_original.shape) * 0.1
for i in range(10000)
])
dataset = hv.Dataset(df)

# Build selection linking object
selection_linker = hv.selection.link_selections.instance()

scatter = selection_linker(
hv.operation.datashader.datashade(
    hv.Scatter(dataset, kdims=["sepal_length"], vdims=["sepal_width"])
)
).opts(title="Datashader with %d points" % len(dataset))

hist = selection_linker(
hv.operation.histogram(dataset, dimension="petal_width", normed=False)
)

# Use plot hook to set the default drag mode to vertical box selection
def set_hist_dragmode(plot, element):
fig = plot.state
fig['layout']['dragmode'] = "select"
fig['layout']['selectdirection'] = "h"

hist.opts(opts.Histogram(hooks=[set_hist_dragmode]))

app = Dash()
components = to_dash(
app, [scatter, hist], reset_button=True
)

app.layout = html.Div(components.children)

if __name__ == "__main__":
    app.run(debug=True)
```

![Demo Of Datashader with 1500000 points and linked selection](/assets/images/integration/holoviews/datashader_linked.gif "")

## Map Overlay
Most 2-dimensional HoloViews elements can be displayed on top of a map by
overlaying them on top of a `Tiles` element.  There are three approaches to
configuring the map that is displayed by a `Tiles` element:

1. Construct a `holoviews.Tiles` element with a templated WMTS tile server url.
 E.g. `Tiles("https://maps.wikimedia.org/osm-intl/{Z}/{X}/{Y}@2x.png")`
2. Construct a `Tiles` element with a predefined tile server url using a
 function from the `holoviews.element.tiles.tile_sources` module. E.g. `CartoDark()`
3. Construct a `holoviews.Tiles` element no constructor argument, then use `.opts`
 to supply a mapbox token and style.
 E.g. `Tiles().opts(mapboxstyle="light", accesstoken="pk...")`

> **Coordinate Systems:**  Unlike the native plotly mapbox traces which accept
geographic positions in longitude and latitude coordinates, HoloViews expects
geographic positions to be supplied in Web Mercator coordinates
(https://epsg.io/3857).  Rather than "longitude" and "latitude", horizontal and
vertical coordinates in Web Mercator are commonly called "easting" and "northing"
respectively. HoloViews provides `Tiles.lon_lat_to_easting_northing` and
`Tiles.easting_northing_to_lon_lat` for converting between coordinate systems.

This example displays a scatter plot of the `plotly.data.carshare` dataset on top
of the predefined `StamenTerrain` WMTS map.

{{example mapbox-points}}

## Visualizing a Large Geographic Dataset with Datashader
This example demonstrates how to use datashader to display a large geographic
scatter plot on top of a vector tiled Mapbox map. A large dataset is synthesized
by repeating the carshare dataset and adding Gaussian noise to the positional
coordinates.

Using mapbox maps requires a free mapbox account and an associated access token.

> This example assumes the mapbox access token is stored in a local file named
`.mapbox_token`. To run this example yourself, either create a file with this name
in your working directory that contains your token, or assign the `mapbox_token`
variable to a string containing your token.

```python
from dash import Dash, html
import holoviews as hv
from holoviews.plotting.plotly.dash import to_dash
from holoviews.operation.datashader import datashade
import pandas as pd
import numpy as np
from plotly.data import carshare
from plotly.colors import sequential

# Mapbox token (replace with your own token string)
mapbox_token = open(".mapbox_token").read()

# Convert from lon/lat to web-mercator easting/northing coordinates
df_original = carshare()
df_original["easting"], df_original["northing"] = hv.Tiles.lon_lat_to_easting_northing(
df_original["centroid_lon"], df_original["centroid_lat"]
)

# Duplicate carshare dataframe with Gaussian noise to produce a larger dataframe
df = pd.concat([df_original] * 5000)
df["easting"] = df["easting"] + np.random.randn(len(df)) * 400
df["northing"] = df["northing"] + np.random.randn(len(df)) * 400

# Build Dataset and graphical elements
dataset = hv.Dataset(df)
points = hv.Points(
df, ["easting", "northing"]
).opts(color="crimson")
tiles = hv.Tiles().opts(mapboxstyle="light", accesstoken=mapbox_token)
overlay = tiles * datashade(points, cmap=sequential.Plasma)
overlay.opts(
title="Mapbox Datashader with %d points" % len(df),
width=800,
height=500
)

# Build App
app = Dash()
components = to_dash(app, [overlay], reset_button=True)

app.layout = html.Div(
components.children
)

if __name__ == '__main__':
    app.run(debug=True)
```

![Demo Of Mapbox DataShader with 1235000 points](/assets/images/integration/holoviews/mapbox_datashader.gif "")

## Mapbox datashader and linked selections
This example demonstrates how the `link_selections` transformation described above
can be used with geographic scatter plots.  Here, the scatter plot is also
datashaded, but `link_selections` will work with plain `Scatter` elements
as well.

```python
from dash import Dash, html
import holoviews as hv
from holoviews.plotting.plotly.dash import to_dash
from holoviews.operation.datashader import datashade
import pandas as pd
import numpy as np
from plotly.data import carshare
from plotly.colors import sequential

# Mapbox token (replace with your own token string)
mapbox_token = open(".mapbox_token").read()

# Convert from lon/lat to web-mercator easting/northing coordinates
df_original = carshare()
df_original["easting"], df_original["northing"] = hv.Tiles.lon_lat_to_easting_northing(
    df_original["centroid_lon"], df_original["centroid_lat"]
)

# Duplicate carshare dataframe with noise to produce a larger dataframe
df = pd.concat([df_original] * 5000)
df["easting"] = df["easting"] + np.random.randn(len(df)) * 400
df["northing"] = df["northing"] + np.random.randn(len(df)) * 400

# Build HoloViews Dataset and visual elements
dataset = hv.Dataset(df).redim.label(car_hours="Car Hours")
points = hv.Points(
    df, ["easting", "northing"]
).opts(color="crimson")
tiles = hv.Tiles().opts(mapboxstyle="light", accesstoken=mapbox_token)
overlay = tiles * datashade(points, cmap=sequential.Plasma).opts(width=800)

# Build histogram of car_hours column
hist = hv.operation.histogram(dataset, dimension="car_hours", normed=False)

# Use plot hook to set the default drag mode to horizontal box selection
def set_hist_dragmode(plot, element):
    fig = plot.state
    fig['layout']['dragmode'] = "select"
    fig['layout']['selectdirection'] = "h"

hist.opts(hooks=[set_hist_dragmode], color="rebeccapurple", margins=(50, 50, 50, 30))

# Link selections across datashaded points and histogram
selection_linker = hv.selection.link_selections.instance()
linked_points = selection_linker(
    tiles * datashade(points, cmap=sequential.Plasma)
).opts(title="Datashader with %d points" % len(dataset), margins=(50, 0, 50, 30))
linked_hist = selection_linker(hist)

# Build app
app = Dash()
components = to_dash(app, [linked_points, linked_hist], reset_button=True)

app.layout = html.Div(
    components.children
)

if __name__ == '__main__':
    app.run(debug=True)
```
![Demo Of Mapbox DataShader and linked selection with 1235000 points](/assets/images/integration/holoviews/mapbox_linked_datashader.gif "")

### GPU Accelerating Datashader and Linked Selections with RAPIDS

Many HoloViews operations, including `datashade` and `link_selections`, can be
accelerated on modern NVIDIA GPUs using technologies from the
[RAPIDS](https://developer.nvidia.com/rapids)
ecosystem. All of the previous examples can be GPU accelerated simply by
replacing the pandas DataFrame passed to the `Dataset` constructor with a
[`cuDF`](https://github.com/rapidsai/cudf) DataFrame.

[![Rapids Logo](/assets/images/integration/RAPIDS-logo-white.svg)](https://developer.nvidia.com/rapids)

The `cudf.from_pandas` function can be used to construct a cuDF DataFrame from a
pandas DataFrame. So adding GPU acceleration to each of the prior examples can be
done simply by replacing the `dataset = hv.Dataset(df)` statements with:

```python
import cudf
dataset = hv.Dataset(cudf.from_pandas(df))
```

## Advanced HoloViews
While motivated by Datashader and linked selections use cases, the
`to_dash` transformation supports arbitrary HoloViews objects and has
full support for the elements and stream types supported by the HoloViews Plotly
backend.

To demonstrate this, here are Dash ports of some of the interactive Plotly examples
from the HoloViews documentation.

### Bounds & selection stream example
Based on
https://holoviews.org/reference/streams/plotly/Bounds.html#streams-plotly-gallery-bounds

> A linked streams example demonstrating how to use Bounds and Selection
streams together.

{{example bounds-stream}}

### BoundsX stream example
Based on
https://holoviews.org/reference/streams/plotly/BoundsX.html#streams-plotly-gallery-boundsx

> A linked streams example demonstrating how to use BoundsX streams.

{{example boundsx-stream}}

### BoundsY stream example
Based on
https://holoviews.org/reference/streams/plotly/BoundsY.html#streams-plotly-gallery-boundsy

> A linked streams example demonstrating how to use BoundsY streams.

{{example boundsy-stream}}

### RangeXY stream example
Based on
https://holoviews.org/reference/streams/plotly/RangeXY.html#streams-plotly-gallery-rangexy

> A linked streams example demonstrating how to use multiple Selection1D
streams on separate Points objects.

{{example rangexy-stream}}

### Multiple selection streams example
Based on
https://holoviews.org/reference/streams/plotly/Selection1D_paired.html#streams-plotly-gallery-selection1d-paired

> A linked streams example demonstrating how to use multiple Selection1D streams
on separate Points objects.

{{example selection1d-paired-stream}}

### Point Selection1D stream example
Based on
https://holoviews.org/reference/streams/plotly/Selection1D_points.html#streams-plotly-gallery-selection1d-points

> A linked streams example demonstrating how to use Selection1D to get
currently selected points and dynamically compute statistics of selection.

{{example selection1d-points-stream}}

### DynamicMap Container
Based on
https://holoviews.org/reference/containers/plotly/DynamicMap.html#containers-plotly-gallery-dynamicmap

> A DynamicMap is an explorable multi-dimensional wrapper around a callable
that returns HoloViews objects.

{{example dynamic-map}}
