---
url: /cytoscape/biopython
name: Biopython Examples
description: "Examples of applications in bioinformatics using Biopython."
languages:
    - python
---

# ![](/assets/images/language_icons/matlab_50px.svg) Cytoscape with Biopython

In this chapter, we will show an example of automatically generating a
phylogenetic tree
from biopython's `Phylo` object, and enable interactive features such
as highlighting children clades. The source code is available as [`usage-phylogeny.py`](https://github.com/plotly/dash-cytoscape/blob/master/demos/usage-phylogeny.py),
and you can find an [online demo here](https://dash.gallery/dash-cytoscape-phylogeny/).

## Parsing the Phylo object

The first step is to load the phylogeny tree. We will be using the
`apaf.xml` file retrieved from [PhyloXML](http://www.phyloxml.org/examples/apaf.xml).

To load the file, run this (after you made sure that
biopython was correctly installed):

```
from Bio import Phylo
tree = Phylo.read('data/apaf.xml', 'phyloxml')
```

Then, we need to use a function to parse the data. We will construct a
function `generate_elements(...)`, which will first generate the column
position and row position of each element in the tree using helper functions
`get_col_positions` and `get_row_positions`, and recursively parse the
tree and add the clades to the list of elements, using a helper function
called `add_to_elements`.


}}


}}


}}

> You might notice that we use something called support clades. Those are
> simply used to modify the shape of the tree so that it resembles
> traditional phylogeny tree layouts.

Finally, we finish building `generate_elements` with the following code:

```
import math

def generate_elements(tree, xlen=30, ylen=30, grabbable=False):
    def col_positions(tree, column_width=80):
        ...

    def row_positions(tree):
        ...

    def add_to_elements(clade, clade_id):
        ...

    col_positions = get_col_positions(tree)
    row_positions = get_row_positions(tree)

    nodes = []
    edges = []

    add_to_elements(tree.clade, 'r')

    return nodes, edges
```

> Note that `add_to_elements` changes the `nodes` and `edges` lists in place.

## Defining layout and stylesheet

Since we are assigning a position to the nodes, we have to use the `preset`
layout. Additionally, we need to add specific styles in order to make the
phylogeny trees to match aesthetically the traditional methods. We define:

```
layout = {'name': 'preset'}

stylesheet = [
    {
        'selector': '.nonterminal',
        'style': {
            'label': 'data(confidence)',
            'background-opacity': 0,
            "text-halign": "left",
            "text-valign": "top",
        }
    },
    {
        'selector': '.support',
        'style': {'background-opacity': 0}
    },
    {
        'selector': 'edge',
        'style': {
            "source-endpoint": "inside-to-node",
            "target-endpoint": "inside-to-node",
        }
    },
    {
        'selector': '.terminal',
        'style': {
            'label': 'data(name)',
            'width': 10,
            'height': 10,
            "text-valign": "center",
            "text-halign": "right",
            'background-color': '#222222'
        }
    }
]
```

## Layout and Callbacks

At this point, we simply need to create the layout of the app, which will
be a simple Cytoscape component. We will also add a callback that will
color all the children of an edge red whenever we hover on that edge. This
is accomplished by filtering the IDs, which are sequences of
*s*'s and *c*'s, which stand for *support* and *children*, intersected
by the number 0 or 1, since there are two subclades per clade.

```
# Start the app
from dash import Dash, html
import dash_cytoscape as cyto

app = Dash()

app.layout = html.Div([
    cyto.Cytoscape(
        id='cytoscape-usage-phylogeny',
        elements=elements,
        stylesheet=stylesheet,
        layout=layout,
        style={
            'height': '95vh',
            'width': '100%'
        }
    )
])

@callback(Output('cytoscape-usage-phylogeny', 'stylesheet'),
              Input('cytoscape-usage-phylogeny', 'mouseoverEdgeData'))
def color_children(edgeData):
    if not edgeData:
        return stylesheet

    if 's' in edgeData['source']:
        val = edgeData['source'].split('s')[0]
    else:
        val = edgeData['source']

    children_style = [{
        'selector': 'edge[source *= "{}"]'.format(val),
        'style': {
            'line-color': 'blue'
        }
    }]

    return stylesheet + children_style
```

This results in the following app:

{{example usage-phylogeny}}

