---
url: /cytoscape/styling
name: Styling
description: "Methods to style elements with a CSS-like syntax."
languages:
    - python
    - julia
---

# ![](/assets/images/language_icons/matlab_50px.svg) Cytoscape Styling

## The `stylesheet` parameter

Just like how the `elements` parameter takes as an input a list of
dictionaries specifying all the elements in the graph, the stylesheet takes
a list of dictionaries specifying the style for a group of elements, a
class of elements, or a single element. Each of these dictionaries accept
two keys:
- `'selector'`: A string indicating which elements you are styling.
- `'style'`: A dictionary specifying what exactly you want to modify. This
could be the width, height, color of a node, the shape of the arrow on an
edge, or many more.

Both [the selector string](http://js.cytoscape.org/#selectors) and
[the style dictionary](http://js.cytoscape.org/#style/node-body) are
exhaustively documented in the Cytoscape.js docs. We will show examples
that can be easily modified for any type of design, and you can create
your own styles with the [online style editor](https://dash-gallery.plotly.host/cytoscape-advanced).

## Basic selectors and styles

We start by looking at the same example shown in the elements
chapter, but this time we examine the stylesheet:


||julia
{{details
View simple elements
```julia
    simple_elements = [
        Dict(
            "data" =>  Dict("id" =>  "one", "label" =>  "Modified Color"),
            "position" =>  Dict("x" =>  75, "y" =>  75),
            "classes" =>  "red" # Single class
        ),
        Dict(
            "data" =>  Dict("id" =>  "two", "label" =>  "Modified Shape"),
            "position" =>  Dict("x" =>  75, "y" =>  200),
            "classes" =>  "triangle" # Single class
        ),
        Dict(
            "data" =>  Dict("id" =>  "three", "label" =>  "Both Modified"),
            "position" =>  Dict("x" =>  200, "y" =>  75),
            "classes" =>  "red triangle" # Multiple classes
        ),
        Dict(
            "data" =>  Dict("id" =>  "four", "label" =>  "Regular"),
            "position" =>  Dict("x" =>  200, "y" =>  200)
        ),
        Dict("data" =>  Dict("source" =>  "one", "target" =>  "two"), "classes" =>  "red"),
        Dict("data" =>  Dict("source" =>  "two", "target" =>  "three")),
        Dict("data" =>  Dict("source" =>  "three", "target" =>  "four"), "classes" =>  "red"),
        Dict("data" =>  Dict("source" =>  "two", "target" =>  "four")),
    ]
```
}}
}}

{{example group_and_class_selectors}}

In this example, we use the group and class selectors. Group selectors
consist of either the string  or the string , since an
element can only be one or the other.

To select a class, you simply pass `.className` to the selector, where
`className` is the name of one of the classes you assigned to some of your
elements. Notice in the example above that if an element has two or more
classes, it will accept all the styles that select it.

Here, we are simultaneously modifying the background and line color of all
the elements of class "red". This means that if the element is a node, then
it will be filled with red, and it is an edge, then the color of the line
will be red. Although this offers great flexibility, you need to be careful
with your declaration, since you won't receive warning if you use the wrong
key or make a typo. Standard RGB and Hex colors are accepted, along with
basic colors recognized by CSS.

Additionally, the `content` key takes as value what you want to display
above or next to the element on the screen, which in this case is the
label inside the `data` dictionary of the input element. Since we defined
a label for each element, that label will be displayed for every node.

## Comparing data items using selectors

A nice property of the selector is that it can select elements by comparing
a certain item of the data dictionaries with a given value. Say we have
some nodes with `id` A to E declared this way:



where the `'weight'` key indicates the weight of your edge. You can find
the declaration below:


||julia
{{details
View weighted elements
```julia
    weighted_elements = [
        Dict("data" =>  Dict("id" =>  "A")),
        Dict("data" =>  Dict("id" =>  "B")),
        Dict("data" =>  Dict("id" =>  "C")),
        Dict("data" =>  Dict("id" =>  "D")),
        Dict("data" =>  Dict("id" =>  "E")),
    
        Dict("data" =>  Dict("source" =>  "A", "target" =>  "B", "weight" =>  1)),
        Dict("data" =>  Dict("source" =>  "A", "target" =>  "C", "weight" =>  2)),
        Dict("data" =>  Dict("source" =>  "B", "target" =>  "D", "weight" =>  3)),
        Dict("data" =>  Dict("source" =>  "B", "target" =>  "E", "weight" =>  4)),
        Dict("data" =>  Dict("source" =>  "C", "target" =>  "E", "weight" =>  5)),
        Dict("data" =>  Dict("source" =>  "D", "target" =>  "A", "weight" =>  6))
    ]
```
}}
}}

If you want to highlight all the of the edges above a certain weight
(e.g. 3), use the selector `'[weight > 3]'`. For example:

{{example weight_selector}}

Similarly, if you want to have weights smaller or equal to 3, you would
write:

{{example weight_selector2}}

Comparisons also work for string matching problems. Given the same graph
as before, but with a data key `'firstname'` for each node:



We can select all the elements that match a specific pattern. For instance,
to style nodes where `'firstname'` contains the substring `'ert'`, we
declare:


||julia
{{details
View named elements
```julia
    named_elements = [
        Dict("data" =>  Dict("id" =>  "A", "firstname" =>  "Albert")),
        Dict("data" =>  Dict("id" =>  "B", "firstname" =>  "Bert")),
        Dict("data" =>  Dict("id" =>  "C", "firstname" =>  "Charlie")),
        Dict("data" =>  Dict("id" =>  "D", "firstname" =>  "Daniela")),
        Dict("data" =>  Dict("id" =>  "E", "firstname" =>  "Emma")),

        Dict("data" =>  Dict("source" =>  "A", "target" =>  "B")),
        Dict("data" =>  Dict("source" =>  "A", "target" =>  "C")),
        Dict("data" =>  Dict("source" =>  "B", "target" =>  "D")),
        Dict("data" =>  Dict("source" =>  "B", "target" =>  "E")),
        Dict("data" =>  Dict("source" =>  "C", "target" =>  "E")),
        Dict("data" =>  Dict("source" =>  "D", "target" =>  "A"))
    ]
```
}}
}}

{{example rectangle_shape}}

Now, if we want to select all the elements where `'firstname'` *does not*
contain `'ert'`, then we can run:

{{example select_elements}}

Other options also exist for matching specific parts of the string. For
example, if we want to only select the prefix, we can use `^=` as such:

{{example select_prefix}}

This can also be prepended by modifiers. For example, `@` added in front
of an operator will render the string matched case insensitive.

{{example case_insensitive}}

View the [complete list of matching operations](http://js.cytoscape.org/#selectors/data)
 for data selectors.

## Styling edges

### Two-sided edges and curvature

Many methods exist to style edges in Dash Cytoscape. In some cases, you
might want different ways to display double-edged


||julia
{{details
View double-edged elements
```julia
    double_edges = [
        Dict("data" =>  Dict("id" =>  string(src,tgt), "source" =>  src, "target" =>  tgt))
        for (src, tgt) in ["AB", "BA", "BC", "CB", "CD", "DC", "DA", "AD"]
    ]

    double_edged_el = vcat([("data" =>  ("id" =>  id_)) for id_ in "ABCD"],double_edges)
```
}}
}}

{{example display_double_edged}}

Many curve styles are accepted, and support further customization such as
the distance between edges and curvature radius. You can find them in
the [JavaScript docs](http://js.cytoscape.org/#style/bezier-edges).

### Edge Arrows

To better highlight the directed edges, we can add arrows of different
shapes, colors, and positions onto the edges. This is an example of using
different types of arrows, with the same graph above, but with certain
edges removed:


||julia
{{details
View directed elements
```julia
    directed_edges = [
        Dict("data" =>  Dict("id" =>  string(src,tgt), "source" =>  src, "target" =>  tgt))
        for (src, tgt) in ["BA", "BC", "CD", "DA"]
    ]

    directed_elements = vcat([Dict("data" =>  ("id" =>  id_)) for id_ in "ABCD"], directed_edges)
```
}}
}}

{{example edge_arrows}}

Notice here that we prepend a position indicator for the color and shape
keys. In the previous example, all four possible positions are displayed.
In fact, you can even the edges with multiple arrows, all with different
colors and shapes. View the [complete list of edge arrow configurations](http://js.cytoscape.org/#style/edge-arrow).

## Displaying Images

It is possible to [display images inside nodes](http://js.cytoscape.org/#style/background-image),
as documented in Cytoscape.js. We show below a complete example of display
an interactive tree of animal phylogeny using images from Wikimedia.

{{example images}}
