---
url: /cytoscape/layout
name: Layouts
description: "Description of built-in layouts, and how to modify their properties."
languages:
    - python
    - julia
---

# ![](/assets/images/language_icons/julia_50px.svg) Cytoscape Layouts

The layout parameter of `cyto.Cytoscape` takes as argument a
dictionary specifying how the nodes should be positioned on the screen.
Every graph requires this dictionary with a value specified for the
`name` key. It represents a built-in display method, which is one of the
following:
- `preset`
- `random`
- `grid`
- `circle`
- `concentric`
- `breadthfirst`
- `cose`

All those layouts, along with their options, are described in the
[official Cytoscape documentation](http://js.cytoscape.org/#layouts).
There, you can find the exact keys accepted by your dictionary, enabling
advanced fine-tuning (demonstrated below).

If preset is given, the positions will be rendered based on the positions
specified in the elements. Otherwise, the positions will be computed by
Cytoscape.js behind the scene, based on the given items of the layout
dictionary. Let's start with an example of declaring a graph with a preset
layout:

{{example preset_layout}}

> Here, we provided toy elements using geographically positioned nodes. If
> you'd like to reproduce this example by yourself, check out the code
> below.


||julia
{{details
View Elements Declaration
```julia
nodes = [
    Dict(
        "data" =>  Dict("id" =>  short, "label" =>  label),
        "position" =>  Dict("x" =>  20*lat, "y" =>  -20*long)
    )
    for (short, label, long, lat) in (
        ("la", "Los Angeles", 34.03, -118.25),
        ("nyc", "New York", 40.71, -74),
        ("to", "Toronto", 43.65, -79.38),
        ("mtl", "Montreal", 45.50, -73.57),
        ("van", "Vancouver", 49.28, -123.12),
        ("chi", "Chicago", 41.88, -87.63),
        ("bos", "Boston", 42.36, -71.06),
        ("hou", "Houston", 29.76, -95.37)
    )
]

edges = [
    Dict("data" =>  Dict("source" =>  source, "target" =>  target))
    for (source, target) in (
        ("van", "la"),
        ("la", "chi"),
        ("hou", "chi"),
        ("to", "mtl"),
        ("mtl", "bos"),
        ("nyc", "bos"),
        ("to", "hou"),
        ("to", "nyc"),
        ("la", "nyc"),
        ("nyc", "bos")
    )
]

elements = vcat(nodes, edges)
```
}}
}}

## Display Methods

In most cases, the position of the nodes will not be given. In these
cases, one of the built-in methods can be used. Let's see what happens
when the value of `name` is set to `'circle'` or `'grid'`

{{example layout_circle}}

{{example layout_grid}}

## Fine-tuning the Layouts

For any given `name` item, a collection of keys are accepted by the layout
dictionary. For example, the
[`grid` layout](http://js.cytoscape.org/#layouts/grid)
will accept `row` and `cols`, the
[`circle` layout](http://js.cytoscape.org/#layouts/circle) `radius`
and `startAngle`, and so forth. Here is the grid layout
with the same graph as above, but with different layout options:

{{example layout_grid_options}}

In the case of the circle layout, we can force the nodes to start and end
at a certain angle in radians:

{{example layout_circle_angles}}

For the `breadthfirst` layout, a tree is created from the existing nodes
by performing a breadth-first search of the graph. By default, the root(s)
of the tree is inferred, but can also be specified as an option. Here is
how the graph would look like if we choose New York City as the root:

{{example layout_breadthfirst}}

Here is what would happen if we chose Montreal and Vancouver instead:

{{example layout_breadthfirst_2}}

> Notice here that we are not giving the ID of the nodes to the `roots`
> key, but instead using a specific syntax to select the desired elements.
> This concept of [selector is extensively documented in Cytoscape.js](http://js.cytoscape.org/#selectors),
> and will be further explored in <dccLink href="/cytoscape/styling" children="part 3"/>.
> We follow the same syntax as the Javascript library.

For preset layouts, you can also specify the positions for which you would like to render each
of your nodes:

{{example layout_positions}}

> In the callbacks chapter, you will learn how to interactively update your layout; in order
> to use `preset`, you will need to specify the position of each node.

## Physics-based Layouts

Additionally, the `cose` layout can be used to position the nodes using
a force-directed layout by simulating attraction and repulsion among the
elements, based on the paper by
[Dogrusoz et al, 2009](https://dl.acm.org/citation.cfm?id=1498047).

{{example layout_cose}}



