---
url: /clientside-callbacks
name: Clientside Callbacks
description:
    "You might want to execute a callback in the frontend as opposed to the backend if you want to avoid the extra time that it takes to make a roundtrip to the server. Clientside callbacks allow you to write your callbacks in JavaScript that runs in the browser."
icon: user.svg
languages:
  - python
  - fsharp
---

# ![](/assets/images/language_icons/f-sharp_50px.svg) Clientside Callbacks

> To get the most out of this page, make sure you've read about <dccLink href="/basic-callbacks" children="Basic Callbacks"/> in the Dash Fundamentals.

Sometimes callbacks can incur a significant overhead, especially when they:
- receive and/or return very large quantities of data (transfer time)
- are called very often (network latency, queuing, handshake)
- are part of a callback chain that requires multiple roundtrips between the browser and Dash.NET

When the overhead cost of a callback becomes too great and no other optimization is possible, the callback can be modified to be run
directly in the browser instead of a making a request to Dash.NET.

The syntax for the callback is almost exactly the same; you use
`Input` and `Output` as you normally would when declaring a callback,
but you also define a JavaScript function as the first argument to the
.

For example, the following callback:



***

Can be rewritten to use JavaScript like so:


***

You also have the option of defining the function in a `.js` file in
your `assets/` folder. To achieve the same result as the code above,
the contents of the `.js` file would look like this:

```javascript
window.dash_clientside = Object.assign({}, window.dash_clientside, {
    clientside: {
        large_params_function: function(largeValue1, largeValue2) {
            return someTransform(largeValue1, largeValue2);
        }
    }
});
```

***

In Dash.NET, the callback would now be written as:

```fsharp
let callback =
    Callback.singleOut (
        [ "in-component1" @. Value
          "in-component2" @. Value ],
        "out-component" @. Value,
        , (fun (_: obj) (_: string) -> CallbackResultBinding.create (Dependency.create(null, null)) null)
        , ClientSideFunction = {
            Namespace = "clientside"
            FunctionName = "large_params_function"
        },
        PreventInitialCall = false
    )
```
The function (`fun (_: obj) (_:string)...`) is only to facilitate type inference and will not be called.


***

## A Simple Example
Below is an example of using clientside callbacks to update a
graph in conjunction with a `Store` component. In the example, we update a `Store`
component on the backend; to create and display the graph, we have a clientside callback in the
frontend that adds some extra information about the layout that we
specify using the radio buttons under "Graph scale".

First, create a JavaScript file `clientside.js` in `WebRoot\assets` with the following content:
```javascript
window.dash_clientside = Object.assign({}, window.dash_clientside, {
    figure: {
        set_scale: function(figure, scale) {
            if(figure === undefined) {
                return {'data': [], 'layout': {} };
            }
            const fig = Object.assign({}, figure, {
                'layout': {
                    ...figure.layout,
                    'yaxis': {
                        ...figure.layout.yaxis, type: scale
                    }
                 }
            });
            return fig;
        }
    }
});
```

{{example graph_update_fe_be_px}}


Note that, in this example, we are manually creating the `figure`
 by extracting the relevant data from the
CSV. This is what gets stored in our
`Store` component;
expand the "Contents of figure storage" above to see exactly what
is used to construct the graph.
Notice a `layout` is already defined. So, we have to mutate the
existing layout in our JavaScript code.



## Using Plotly Express to Generate a Figure



{{example graph_update_fe_be_px}}

Again, you can expand the "Contents of figure storage" section
above to see what gets generated. You may notice that this is
quite a bit more extensive than the previous example; in
particular, a `layout` is already defined. So, instead of creating
a `layout` as we did previously, we have to mutate the existing
layout in our JavaScript code.

## Clientside Callbacks with Promises

Dash 2.4 and later supports clientside callbacks that return promises.

### Fetching Data Example

In this example, we fetch data (based on the value of the dropdown) using an async clientside callback function that outputs it to a `dash_table.DataTable` component.

{{example promise}}

### Notifications Example

This example uses promises and sends desktop notifications to the user once they grant permission and select the **Notify** button:

{{example promise_2}}

![Notification with promises](/assets/images/clientside-callbacks/notification.gif)

***

## Callback Context

You can use `dash_clientside.callback_context.triggered_id` within a clientside callback to access the ID of the component that triggered the callback.

In this example, we display the `triggered_id` in the app when a button is clicked.

{{example cb_context_triggered}}

## Set Props

*New in 2.16*

`dash_clientside.set_props` allows you to update a Dash component property directly instead of updating it by having it as an output of a clientside callback. This can be useful if you have a non-Dash component (for example, a custom JavaScript component) that you want to update a Dash component property from, or if you want to implement custom functionality that is not available directly within Dash but that interacts with Dash.


> 


The following example adds an event listener to the page. This event listener responds to the user pressing <kbd>Ctrl</kbd>+<kbd>R</kbd> by updating a `dcc.Store` component's `data`. Another callback has the `dcc.Store` component's `data` property as an input so runs each time it changes, outputting the updated `data` to an `html.Div` component.

{{example set_props_example}}

**Notes about this example**

- `dash_clientside.set_props` takes two arguments. The first is the ID of the Dash component to update. The second is an object with the name of the property to update as a key, and the value as the new value to update that property to. In this example `dash_clientside.set_props("store-events", {data: newData})` updates the `data` property of the Dash component with ID `store-events`, with a new value of `newData`, which here is a variable that contains a string representation of the current date and time.
- The clientside callback returns `dash_clientside.no_update`, meaning it doesn't update any Dash component specified as an `Output`. The only update that happens to the page from the clientside callback is via `dash_clientside.set_props`.
- The `Input` for the callback that adds the event listener is the ID of the app's main container `html.Div`. We use the `id` property as this won't change after our app loads, meaning this clientside callback only runs when the app loads.
- In this example, the `Output` is the same as the `Input`, but it could be anything because we don't update the `Output`.

## Limitations

There are a few limitations to keep in mind:

1. Clientside callbacks execute on the browser's main thread and will block
rendering and events processing while being executed.
2. Clientside callbacks are not possible if you need to refer to global
variables on the server or a DB call is required.
3. Dash.NET does not support asynchronous clientside callbacks and will
fail if a `Promise` is returned.

